//-------------------------------------------------------------------------------------------------------------------------------------------------------------
//
// Copyright 2023 Apple Inc.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
//
//-------------------------------------------------------------------------------------------------------------------------------------------------------------

#import "GameViewController.h"
#import "RenderBridge.h"

@implementation GameViewController
{
    MTKView *_view;
    RenderBridge *_renderer;
}

#pragma region - Command-line configuration
- (void)evaluateCommandLine
{
    NSArray<NSString *>* args = [[NSProcessInfo processInfo] arguments];
    BOOL exitAfterOneFrame = [args containsObject:@"--auto-close"];
    
    if (exitAfterOneFrame)
    {
        NSLog(@"Automatically terminating in 5 seconds...");
        [[NSApplication sharedApplication] performSelector:@selector(terminate:) withObject:self afterDelay:5];
    }
}

#pragma region - UI configuration

- (void)viewDidLoad
{
    [super viewDidLoad];
    [self evaluateCommandLine];
    
    [self _configureBackdrop:_configBackdrop];
    
    _view = (MTKView *)self.view;
    
    _view.device = MTLCreateSystemDefaultDevice();
    _view.colorPixelFormat = MTLPixelFormatBGRA8Unorm_sRGB;
    _view.depthStencilPixelFormat = MTLPixelFormatInvalid; // no depth testing
    _view.clearColor = MTLClearColorMake( 0.53, 0.81, 0.92, 1.0 );
    _view.clearDepth = 1.0f;
    _view.depthStencilStorageMode = MTLStorageModeMemoryless;
    _view.framebufferOnly = NO;
    
    if(!_view.device)
    {
        NSLog(@"Metal is not supported on this device");
        self.view = [[NSView alloc] initWithFrame:self.view.frame];
        return;
    }
    
    // Compile shaders in a secondary high-priority thread to prevent UI stalls:
    [_progressIndicator startAnimation:self];
    
    CGSize boundsSize = _view.bounds.size;
    __block GameViewController* weakSelf = self;
    dispatch_async(dispatch_get_global_queue(DISPATCH_QUEUE_PRIORITY_HIGH, 0), ^(){
        
        GameViewController* strongSelf = weakSelf;
        
        strongSelf->_renderer = [[RenderBridge alloc] initWithMetalKitView:self->_view];
        [strongSelf->_renderer mtkView:self->_view drawableSizeWillChange:boundsSize];
        strongSelf->_view.delegate = self->_renderer;
        
        dispatch_async(dispatch_get_main_queue(), ^(){
            [strongSelf->_progressIndicator stopAnimation:self];
            [strongSelf->_progressIndicator setHidden:YES];
        });
    });
}

- (void)updateWindowTitleAndTargetFPS
{
    NSScreen* screen = _view.window.screen;
    NSString* title = [NSString stringWithFormat:@"Metal Shader Converter DXR 1.1 Example (%@ @ %ldHz)", screen.localizedName, (long)screen.maximumFramesPerSecond];
    _view.window.title = title;
    _view.preferredFramesPerSecond = screen.maximumFramesPerSecond;
    _view.window.delegate = self;
}

- (void)viewDidAppear
{
    [self updateWindowTitleAndTargetFPS];
}

- (void)windowDidChangeScreen:(NSNotification *)notification
{
    [self updateWindowTitleAndTargetFPS];
}

- (void)_configureBackdrop:(NSView *)view
{
    view.wantsLayer = YES;
    view.layer.borderWidth = 1.0f;
    view.layer.cornerRadius = 8.0f;
}

@end
