//-------------------------------------------------------------------------------------------------------------------------------------------------------------
//
// Copyright 2025 Apple Inc.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
//
//-------------------------------------------------------------------------------------------------------------------------------------------------------------
#import <Metal/Metal.h>

NS_ASSUME_NONNULL_BEGIN

#define DXRShaderIdentifierNull 0

/// Compilation options to feed to MetalShaderConverter
@interface DXRPipelineOptions : NSObject
@property(nonatomic) uint32_t maxRecursiveDepth;
@property(nonatomic) uint32_t maxAttributeSizeInBytes;
@end

/// Representation of a DXR shader
@interface DXRShader : NSObject
@property(nonatomic) NSString *name;
@property(nonatomic, nullable) NSString *specializedName;
@property(nonatomic) NSData *DXIL;
- (instancetype)initWithName:(NSString *)name DXIL:(NSData *)DXIL;
- (NSString *)uniqueName;
@end

/// Representation of a DXR hitgroup
@interface DXRHitgroup : NSObject
@property(nonatomic) NSString *name;
@property(nonatomic, nullable) DXRShader *intersectionShader;
@property(nonatomic, nullable) DXRShader *anyHitShader;
@property(nonatomic, nullable) DXRShader *closestHitShader;
- (instancetype)initWithName:(NSString *)name
          intersectionShader:(nullable DXRShader *)intersectionShader
                anyHitShader:(nullable DXRShader *)anyHitShader
            closestHitShader:(nullable DXRShader *)closestHitShader;
@end

/// Describes a DXR pipeline
@interface DXRPipelineDescriptor : NSObject
@property(nonatomic) DXRShader *rayGenerationShader;
@property(nonatomic) NSArray<DXRShader *> *missShaders;
@property(nonatomic, nullable) NSArray<DXRShader *> *callableShaders;
@property(nonatomic, nullable) NSArray<DXRHitgroup *> *hitGroups;
@property(nonatomic, nullable) DXRPipelineOptions *options;
@property(nonatomic, nullable) NSDictionary *globalRootSignatureDescriptor;
@property(nonatomic, nullable) NSDictionary *localRootSignatureDescriptor;
@end

/// Member of the DXRPipelineReflection objects with info about converted DXRShaders
@interface DXRShaderConversion : NSObject
/// DXRShader unique name
@property(nonatomic, readonly) NSString *name;
/// The converted MTLFunction
@property(nonatomic, readonly) id<MTLFunction> function;
/// Flag to determine if the `function` was  materialized by metal shader converter
@property(nonatomic, readonly) BOOL isSynthesized;
@end

/// Pipeline Reflection for a DXRPipelineState
@interface DXRPipelineReflection : NSObject
@property(nonatomic, readonly) NSArray<DXRShaderConversion *> *conversions;
@property(nonatomic, readonly) MTLComputePipelineReflection *computePipelineReflection;
@end

/// Holds the underlying converted compute pipeline state and child function handles and tables
@interface DXRPipelineState : NSObject
/// The metal compute pipeline state that represents the DXR pipeline
@property(nonatomic, readonly) id<MTLComputePipelineState> computePipelineState;
/// `visibleFunctionTable` if non-nil, wiill contain handles for miss and closest-hit shaders and may contain the ray generation shader, intersection shader,  any-hit shader
@property(nonatomic, readonly, nullable) id<MTLVisibleFunctionTable> visibleFunctionTable;
/// `intersectionFunctionTable` if non-nil, will contain synthesized intersection function handles for either triangles, procedural or both
@property(nonatomic, readonly, nullable) id<MTLIntersectionFunctionTable> intersectionFunctionTable;
/// `intersectionFunctionBufferHandles` if non-nil and devices supports MTLGPUFamilyApple9, will contain funciton handles for intersection shader, any-hit shader or both
@property(nonatomic, readonly, nullable) NSArray<id<MTLFunctionHandle>> *intersectionFunctionBufferHandles;
/// Returns shader identifer given DXRShader unique name. `0` identifier is invalid
- (uint64_t)shaderIdentifierForName:(NSString *)name;
/// Returns the synthesized triangle interesection shader identifer if  `intersectionFunctionTable` is non-nil .`0` identifier is invalid.
- (uint64_t)shaderIdentifierForSynthesizedTriangleIntersection;
/// Returns the synthesized procedural interesection shader identifer if  `intersectionFunctionTable` is non-nil .`0` identifier is invalid.
- (uint64_t)shaderIdentifierForSynthesizedProceduralIntersection;
@end

/// Converters a DXR pipeline descriptor into a Metal compute pipeline state
@interface DXRCompiler : NSObject
- (instancetype)initWithDevice:(id<MTLDevice>)device;
- (nullable DXRPipelineState *)newDXRPipelineWithDescriptor:(DXRPipelineDescriptor *)descriptor
                                                 reflection:(DXRPipelineReflection *_Nullable*_Nullable)reflection
                                                      error:(NSError **)error;
@end

NS_ASSUME_NONNULL_END
