//-------------------------------------------------------------------------------------------------------------------------------------------------------------
//
// Copyright 2025 Apple Inc.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
//
//-------------------------------------------------------------------------------------------------------------------------------------------------------------

#import "DXRCompiler.h"
#import <metal_irconverter/metal_irconverter.h>
#if IR_VERSION_MAJOR < 3
#error "Metal Shader Converter 3.0 or later required"
#endif
#define IR_PRIVATE_IMPLEMENTATION // For access to APIs for synthesizing required raytracing shader functions
#include <metal_irconverter_runtime/metal_irconverter_runtime.h>

#pragma mark - Helpers

#define out_assign(PARAM, VAL) do { if (PARAM != nil) { *PARAM = (VAL); } } while (0)
#define default_error(_DESC) make_ns_error(__LINE__, _DESC)

static NSError *make_ns_error(int code, NSString *description)
{
    return [NSError errorWithDomain:@"DXRCompiler" code:code userInfo:@{
        NSLocalizedDescriptionKey: description
    }];
}

// __cleanup__ attribute calls
static void destroy_ir_compiler(IRCompiler **compiler)
{
    IRCompilerDestroy(*compiler);
}

static void destroy_ir_object(IRObject **object)
{
    IRObjectDestroy(*object);
}

static void destroy_ir_rootsig(IRRootSignature **rootSignature)
{
    IRRootSignatureDestroy(*rootSignature);
}

static void destroy_ir_rtconfig(IRRayTracingPipelineConfiguration **config)
{
    IRRayTracingPipelineConfigurationDestroy(*config);
}

static void destroy_ir_metallib(IRMetalLibBinary **metallib)
{
    IRMetalLibBinaryDestroy(*metallib);
}

static IRCompiler *DefaultIRCompilerCreate(IRGPUFamily gpuFamily)
{
    // Since we are compiling in process, determine the deployment target version from the macOS app we are running on.
    // This corresponds to the argument for metal-shaderconverter command option `--minimum-os-build-version` when compiled offline.
    char version[32] = {0};
    NSOperatingSystemVersion os_ver = NSProcessInfo.processInfo.operatingSystemVersion;
    snprintf(version, sizeof(version), "%ld.%ld", os_ver.majorVersion, os_ver.minorVersion);
    
    IRCompiler *compiler = IRCompilerCreate();
    IRCompilerSetMinimumDeploymentTarget(compiler, IROperatingSystem_macOS, version);
    IRCompilerSetMinimumGPUFamily(compiler, gpuFamily);
    
    return compiler;
}

static IRRootSignature *RootSignatureCreateWithJSON(NSDictionary *jsonDescriptor, NSError **error)
{
    NSData *data = [NSJSONSerialization dataWithJSONObject:jsonDescriptor options:NSJSONWritingPrettyPrinted error:error];
    if (data == nil)
    {
        out_assign(error, default_error(@"Invalid JSON root signature descriptor"));
        return NULL;
    }
    
    NSString *json = [[NSString alloc] initWithData:data encoding:NSUTF8StringEncoding];
    IRVersionedRootSignatureDescriptor *rootSigDescriptor = IRVersionedRootSignatureDescriptorCreateFromJSON(json.UTF8String);
    IRError *rootSigError = NULL;
    IRRootSignature *rootSig = IRRootSignatureCreateFromDescriptor(rootSigDescriptor, &rootSigError);
    if (rootSig == NULL)
    {
        NSString *description = [NSString stringWithFormat:@"Failed to create root signature: %s\n", (const char *)IRErrorGetPayload(rootSigError)];
        out_assign(error, default_error(description));
        IRErrorDestroy(rootSigError);
        return NULL;
    }
    
    IRVersionedRootSignatureDescriptorRelease(rootSigDescriptor);
    return rootSig;
}

#pragma mark - DXRCompiler

typedef struct
{
    IRRayGenerationCompilationMode rayGeneration;
    IRIntersectionFunctionCompilationMode intersectionFunction;
} DXRCompilationMode;

/// Holds information about  how you can find a DXR converted MTLFunction from compilation intermediates
@interface DXR2MTL : NSObject
@property (readonly) NSString *name;
@property (readonly) MTLFunctionType functionType;
@property (readonly) NSUInteger locationIndex;
@end
/// Holds temporary objects used to create a compute pipeline state for a DXR pipeline descriptor
@interface DXRIntermediates : NSObject
/// Generated kernel function for dispatch
@property(nonatomic) id<MTLFunction> kernelFunction;
/// Generated visible functions
@property(nonatomic) NSArray<id<MTLFunction>> *visibleFunctions;
/// Generated intersection functions
@property(nonatomic) NSArray<id<MTLFunction>> *intersectionFunctions;

/// `convertedMapping` hold a mapping of DXRShader.uniqueName to converted info about the generated metal function.
/// ```
///    // Find a Metal function for a DXR shader
///    DXR2MTL *mapping = intermediate.convertedMapping[dxrShader.uniqueName];
///    id<MTLFunction> function = nil;
///    if (mapping.functionType == MTLFunctionVisible) {
///        NSUInteger index = mapping.locationIndex;
///        function = intermediate.visibleFunctions[index];
///    }
///```
@property(nonatomic) NSDictionary<NSString *, DXR2MTL *> *convertedMapping;

///`synthesizedMapping` are for metal functions JIT'ed by metal shader converter. This is expected for
///`IRRayGenerationCompilationModeVisibleFunction` or`IRIntersectionFunctionCompilationModeVisibleFunction`
@property(nonatomic, nullable) NSArray<DXR2MTL *> *synthesizedMapping;
@end

@interface DXRShaderConversion ()
- (instancetype)initWithName:(NSString *)name function:(id<MTLFunction>)function isSynthesized:(BOOL)isSynthesized;
@end

@interface DXRPipelineReflection ()
@property(nonatomic) NSArray<DXRShaderConversion *> *conversions;
@property(nonatomic) MTLComputePipelineReflection *computePipelineReflection;
@end

@interface DXRPipelineState ()
@property(nonatomic) id<MTLComputePipelineState> computePipelineState;
@property(nonatomic, nullable) id<MTLVisibleFunctionTable> visibleFunctionTable;
@property(nonatomic, nullable) id<MTLIntersectionFunctionTable> intersectionFunctionTable;
@property(nonatomic, nullable) NSArray<id<MTLFunctionHandle>> *intersectionFunctionBufferHandles;
// Private
@property(nonatomic) NSDictionary<NSString *, NSNumber *> *shaderIdentifiers;
@end

@interface DXR2MTL ()
- (instancetype)initWithName:(NSString *)name type:(MTLFunctionType)type;
- (instancetype)initWithName:(NSString *)name type:(MTLFunctionType)type index:(NSUInteger)index;
@end

@implementation DXRCompiler
{
    id<MTLDevice> _device;
}

- (instancetype)initWithDevice:(id<MTLDevice>)device
{
    self = [super init];
    if (self)
    {
        self->_device = device;
    }
    return self;
}

- (nullable DXRPipelineState *)newDXRPipelineWithDescriptor:(DXRPipelineDescriptor *)descriptor
                                                 reflection:(DXRPipelineReflection **)reflection
                                                      error:(NSError **)error
{
    // Validate the descriptor
    if (![self _validatePipelineDescriptor:descriptor error:error]) {
        return nil;
    }
    
    // Determine the compilation mode for ray generation and intersection functions
    DXRCompilationMode compilationMode = [self _defaultCompilationMode];
    
    // Convert shaders in the descriptor to intermediate metal functions (This can be repl
    DXRIntermediates *intermediates = [self _convertDXRShadersWithDescriptor:descriptor compilationMode:compilationMode error:error];
    if (intermediates == nil) {
        return nil;
    }
    
    // Create the ray tracing metal compute pipeline state with the intermediate objects
    MTLComputePipelineReflection *computePipelineReflection = nil;
    id<MTLComputePipelineState> computePipelineState = [self _createComputePipelineStateWithIntermediates:intermediates
                                                                                               reflection:&computePipelineReflection
                                                                                                    error:error];
    if (computePipelineState == nil) {
        return nil;
    }
    
    // Construct DXR pipeline state object (consisting of metal pipelines and resources)
    DXRPipelineState *dxrPipelineState = [self _createDXRPipelineStateWithComputePipelineState:computePipelineState
                                                                                 intermediates:intermediates
                                                                               compilationMode:compilationMode
                                                                                         error:error];
    if (dxrPipelineState == nil) {
        return nil;
    }
    
    // Optionally, set DXR pipeline reflection
    out_assign(reflection, [self _createReflectionWithComputePipelineReflection:computePipelineReflection intermediates:intermediates]);
    
    return dxrPipelineState;
}

- (BOOL)_validatePipelineDescriptor:(DXRPipelineDescriptor *)descriptor error:(NSError **)error
{
    // Validate shaders have unique names
    NSMutableSet<NSString *> *uniqueNames = [[NSMutableSet alloc] init];
    
    BOOL (^InsertUniqueName)(NSString *, NSError **) = ^BOOL (NSString *uniqueName, NSError **u_error) {
        if ([uniqueNames containsObject:uniqueName])
        {
            NSString *description = [NSString stringWithFormat:@"Duplicate shader name: \"%@\"", uniqueName];
            out_assign(u_error, default_error(description));
            return NO;
        }
        
        [uniqueNames addObject:uniqueName];
        return YES;
    };
    
    if (!InsertUniqueName(descriptor.rayGenerationShader.uniqueName, error)) {
        return NO;
    }
    
    for (DXRShader *missShader in descriptor.missShaders)
    {
        if (!InsertUniqueName(missShader.uniqueName, error)) {
            return NO;
        }
    }
    
    for (DXRShader *callableShader in descriptor.callableShaders)
    {
        if (!InsertUniqueName(callableShader.uniqueName, error)) {
            return NO;
        }
    }
    
    for (DXRHitgroup *hitgroup in descriptor.hitGroups)
    {
        if (hitgroup.intersectionShader && !InsertUniqueName(hitgroup.intersectionShader.uniqueName, error)) {
            return NO;
        }
        
        if (hitgroup.anyHitShader && !InsertUniqueName(hitgroup.anyHitShader.uniqueName, error)) {
            return NO;
        }
        
        if (hitgroup.closestHitShader && !InsertUniqueName(hitgroup.closestHitShader.uniqueName, error)) {
            return NO;
        }
    }
    
    return YES;
}

- (DXRCompilationMode)_defaultCompilationMode
{
    // Defaults
    IRRayGenerationCompilationMode rayGenCompilationMode = IRRayGenerationCompilationKernel;
    IRIntersectionFunctionCompilationMode intersectionCompilationMode = IRIntersectionFunctionCompilationIntersectionFunctionBufferFunction;
    
    // Fallback, if device does not support intersection function buffers
    if (self.gpuFamily < IRGPUFamilyApple9)
    {
        intersectionCompilationMode = IRIntersectionFunctionCompilationVisibleFunction;
    }
    
#if DEBUG && 0
    rayGenCompilationMode = IRRayGenerationCompilationVisibleFunction;
    intersectionCompilationMode = IRIntersectionFunctionCompilationVisibleFunction;
#endif
    
    DXRCompilationMode defaultCompilationMode = {
        .rayGeneration = rayGenCompilationMode,
        .intersectionFunction = intersectionCompilationMode,
    };
    
    return defaultCompilationMode;
}

- (DXRIntermediates *)_convertDXRShadersWithDescriptor:(DXRPipelineDescriptor *)descriptor
                                       compilationMode:(DXRCompilationMode)compilationMode
                                                 error:(NSError **)error
{
    // Default ray tracing configuration for shader conversion
    IRRayGenerationCompilationMode rayGenCompilationMode = compilationMode.rayGeneration;
    IRIntersectionFunctionCompilationMode intersectionCompilationMode = compilationMode.intersectionFunction;
    
    // Check support
    if (intersectionCompilationMode == IRIntersectionFunctionCompilationIntersectionFunction)
    {
        out_assign(error, default_error(@"IRIntersectionFunctionCompilationIntersectionFunction not supported"));
        return nil;
    }
    
    if ((intersectionCompilationMode == IRIntersectionFunctionCompilationIntersectionFunctionBufferFunction) && (self.gpuFamily < IRGPUFamilyApple9))
    {
        out_assign(error, default_error(@"MTLDevice does not support IntersectionFunctionBuffer. MTLGPUFamilyApple9 or later is required"));
        return nil;
    }
    
    // From here, use the metal-shaderconverter API to generate converted MetalLibs and the target MTLFunction objects
    // The MetalLib conversion process show here can be easily reproduced with the metal-shaderconverter CLI.
    // See `metal-shaderconverter --help` for more.
    
    //
    // Create root signatures for pipline
    //
    
    __attribute__((__cleanup__(destroy_ir_rootsig)))
    IRRootSignature *globalRootSignature = NULL;
    
    __attribute__((__cleanup__(destroy_ir_rootsig)))
    IRRootSignature *localRootSignature = NULL;
    {
        if (descriptor.globalRootSignatureDescriptor)
        {
            IRRootSignature *rootSig = RootSignatureCreateWithJSON(descriptor.globalRootSignatureDescriptor, error);
            if (rootSig == NULL)
            {
                return nil;
            }
            globalRootSignature = rootSig;
        }
        
        if (descriptor.localRootSignatureDescriptor)
        {
            IRRootSignature *rootSig = RootSignatureCreateWithJSON(descriptor.localRootSignatureDescriptor, error);
            if (rootSig == NULL)
            {
                return nil;
            }
            localRootSignature = rootSig;
        }
    }
    
    //
    // Set up ray tracing pipeline config
    //
        
    // Override with provided options or use defaults
    DXRPipelineOptions *options = descriptor.options ?: [DXRPipelineOptions new];
    int maxRecursiveDepth = (int)options.maxRecursiveDepth;
    uint32_t maxAttributeSizeInBytes = options.maxAttributeSizeInBytes;
    
    // Determine optimal pipeline flags
    IRRaytracingPipelineFlags pipelineFlags = IRRaytracingPipelineFlagNone;
    {
        NSUInteger hitGroupCount = descriptor.hitGroups.count;
        NSUInteger intersectionShaderCount = 0;
        for (DXRHitgroup *hitGroup in descriptor.hitGroups) {
            intersectionShaderCount += (hitGroup.intersectionShader != nil) ? 1 : 0;
        }
        
        if (intersectionShaderCount == 0) {
            pipelineFlags |= IRRaytracingPipelineFlagSkipProceduralPrimitives;
        }
        
        if (intersectionShaderCount == hitGroupCount) {
            pipelineFlags |= IRRaytracingPipelineFlagSkipTriangles;
        }
    }
    
    // Gather intrinsic usage to help optimize code genration during compilation
    uint64_t maskClosestHit = 0, maskAnyHit = 0, maskMiss = 0, maskCallable = 0;
    {
        for (DXRHitgroup *hitGroup in descriptor.hitGroups)
        {
            DXRShader *closestHitShader = hitGroup.closestHitShader;
            if (closestHitShader)
            {
                IRObject *DXIR = IRObjectCreateFromDXIL(closestHitShader.DXIL.bytes, closestHitShader.DXIL.length, IRBytecodeOwnershipNone);
                maskClosestHit |= IRObjectGatherRaytracingIntrinsics(DXIR, closestHitShader.name.UTF8String);
                IRObjectDestroy(DXIR);
            }
            
            DXRShader *anyHitShader = hitGroup.anyHitShader;
            if (anyHitShader)
            {
                IRObject *DXIR = IRObjectCreateFromDXIL(anyHitShader.DXIL.bytes, anyHitShader.DXIL.length, IRBytecodeOwnershipNone);
                maskAnyHit |= IRObjectGatherRaytracingIntrinsics(DXIR, anyHitShader.name.UTF8String);
                IRObjectDestroy(DXIR);
            }
        }
        
        for (DXRShader *missShader in descriptor.missShaders)
        {
            IRObject *DXIR = IRObjectCreateFromDXIL(missShader.DXIL.bytes, missShader.DXIL.length, IRBytecodeOwnershipNone);
            maskMiss |= IRObjectGatherRaytracingIntrinsics(DXIR, missShader.name.UTF8String);
            IRObjectDestroy(DXIR);
        }
        
        for (DXRShader *callableShader in descriptor.callableShaders)
        {
            IRObject *DXIR = IRObjectCreateFromDXIL(callableShader.DXIL.bytes, callableShader.DXIL.length, IRBytecodeOwnershipNone);
            maskCallable |= IRObjectGatherRaytracingIntrinsics(DXIR, callableShader.name.UTF8String);
            IRObjectDestroy(DXIR);
        }
    }
    
    __attribute__((__cleanup__(destroy_ir_rtconfig)))
    IRRayTracingPipelineConfiguration *config = IRRayTracingPipelineConfigurationCreate();
    {
        IRRayTracingPipelineConfigurationSetMaxAttributeSizeInBytes(config, maxAttributeSizeInBytes);
        IRRayTracingPipelineConfigurationSetIntrinsicMasks(config, maskClosestHit, maskMiss, maskAnyHit, maskCallable);
        IRRayTracingPipelineConfigurationSetRayGenerationCompilationMode(config, rayGenCompilationMode);
        IRRayTracingPipelineConfigurationSetIntersectionFunctionCompilationMode(config, intersectionCompilationMode);
        IRRayTracingPipelineConfigurationSetMaxRecursiveDepth(config, maxRecursiveDepth);
        IRRayTracingPipelineConfigurationSetPipelineFlags(config, pipelineFlags);
    }
    
    //
    // Create compiler and begin DXIL to Metal IR conversion
    //
    
    __attribute__((__cleanup__(destroy_ir_compiler)))
    IRCompiler *compiler = DefaultIRCompilerCreate(self.gpuFamily);
    IRCompilerSetGlobalRootSignature(compiler, globalRootSignature);
    IRCompilerSetLocalRootSignature(compiler, localRootSignature);
    IRCompilerSetRayTracingPipelineConfiguration(compiler, config);
    
    id<MTLFunction> kernelFunction = nil;
    NSMutableArray<id<MTLFunction>> *visibleFunctions = [[NSMutableArray alloc] init];
    NSMutableArray<id<MTLFunction>> *intersectionFunctions = [[NSMutableArray alloc] init];
    NSMutableArray<DXR2MTL *> *synthesizedMapping = [[NSMutableArray alloc] init];
    NSMutableDictionary<NSString *, DXR2MTL *> *convertedMapping = [[NSMutableDictionary alloc] init];
    
    // Ray generation
    {
        DXRShader *rayGenerationShader = descriptor.rayGenerationShader;
        assert(rayGenerationShader != nil);
        
        id<MTLFunction> function = [self _functionWithDXRShader:rayGenerationShader stage:IRShaderStageRayGeneration compiler:compiler error:error];
        if (function == nil)
        {
            return nil;
        }
        
        DXR2MTL *mapping = nil;
        MTLFunctionType functionType = function.functionType; // Function type is determined by IRRayGenerationCompilationMode
        assert((functionType == MTLFunctionTypeKernel) || (functionType == MTLFunctionTypeVisible));
        if (functionType == MTLFunctionTypeKernel)
        {
            assert(rayGenCompilationMode == IRRayGenerationCompilationKernel);
            mapping = [[DXR2MTL alloc] initWithName:function.name type:functionType];
            kernelFunction = function;
        }
        else if (functionType == MTLFunctionTypeVisible)
        {
            assert(rayGenCompilationMode == IRRayGenerationCompilationVisibleFunction);
            mapping = [[DXR2MTL alloc] initWithName:function.name type:functionType index:visibleFunctions.count];
            [visibleFunctions addObject:function];
        }
        
        NSString *dxrShaderName = rayGenerationShader.uniqueName;
        convertedMapping[dxrShaderName] = mapping;
    }
    
    // Hitgroups
    for (NSUInteger hitGroupIndex = 0, hitGroupCount = descriptor.hitGroups.count; hitGroupIndex < hitGroupCount; ++hitGroupIndex)
    {
        DXRHitgroup *hitGroup = descriptor.hitGroups[hitGroupIndex];
        DXRShader *intersectionShader = hitGroup.intersectionShader;
        DXRShader *anyHitShader = hitGroup.anyHitShader;
        DXRShader *closestHitShader = hitGroup.closestHitShader;
        
        IRHitGroupType hitGroupType = (intersectionShader != nil) ? IRHitGroupTypeProceduralPrimitive: IRHitGroupTypeTriangles;
        IRCompilerSetHitgroupType(compiler, hitGroupType);
        
        //
        // If both intersectionShader and anyHitShader are provided, Metal shader converter requires that both shaders be fused into a single one.
        // Otherwise the shaders are compiled exclusively.
        //
        
        // Exclusive Intersection shader
        if (intersectionShader && (anyHitShader == nil))
        {
            id<MTLFunction> function = [self _functionWithDXRShader:intersectionShader stage:IRShaderStageIntersection compiler:compiler error:error];
            if (function == nil)
            {
                return nil;
            }
            
            DXR2MTL *mapping = nil;
            MTLFunctionType functionType = function.functionType; // Function type is determined by IRIntersectionFunctionCompilationMode
            assert((functionType == MTLFunctionTypeVisible) || (functionType == MTLFunctionTypeIntersection));
            if (functionType == MTLFunctionTypeVisible)
            {
                assert(intersectionCompilationMode == IRIntersectionFunctionCompilationVisibleFunction);
                mapping = [[DXR2MTL alloc] initWithName:function.name type:functionType index:visibleFunctions.count];
                [visibleFunctions addObject:function];
            }
            else if (functionType == MTLFunctionTypeIntersection)
            {
                assert(intersectionCompilationMode == IRIntersectionFunctionCompilationIntersectionFunctionBufferFunction);
                mapping = [[DXR2MTL alloc] initWithName:function.name type:functionType index:intersectionFunctions.count];
                [intersectionFunctions addObject:function];
            }
            
            NSString *dxrShaderName = intersectionShader.uniqueName;
            convertedMapping[dxrShaderName] = mapping;
        }
        
        // Exclusive Any-hit shader
        if (anyHitShader && (intersectionShader == nil))
        {
            id<MTLFunction> function = [self _functionWithDXRShader:anyHitShader stage:IRShaderStageAnyHit compiler:compiler error:error];
            if (function == nil)
            {
                return nil;
            }
            
            DXR2MTL *mapping = nil;
            MTLFunctionType functionType = function.functionType; // Function type is determined by IRIntersectionFunctionCompilationMode
            assert((functionType == MTLFunctionTypeVisible) || (functionType == MTLFunctionTypeIntersection));
            if (functionType == MTLFunctionTypeVisible)
            {
                assert(intersectionCompilationMode == IRIntersectionFunctionCompilationVisibleFunction);
                mapping = [[DXR2MTL alloc] initWithName:function.name type:functionType index:visibleFunctions.count];
                [visibleFunctions addObject:function];
            }
            else
            {
                assert(intersectionCompilationMode == IRIntersectionFunctionCompilationIntersectionFunctionBufferFunction);
                mapping = [[DXR2MTL alloc] initWithName:function.name type:functionType index:intersectionFunctions.count];
                [intersectionFunctions addObject:function];
            }
            
            NSString *dxrShaderName = anyHitShader.uniqueName;
            convertedMapping[dxrShaderName] = mapping;
        }
        
        // Fused Intersection and Any-hit shader
        if (intersectionShader && anyHitShader)
        {
            id<MTLFunction> function = [self _functionWithDXRAnyHitShader:anyHitShader
                                             fuseedWithIntersectionShader:intersectionShader
                                                                 compiler:compiler error:error];
            if (function == nil)
            {
                return nil;
            }
            
            DXR2MTL *mapping = nil;
            MTLFunctionType functionType = function.functionType;
            assert((functionType == MTLFunctionTypeVisible) || (functionType == MTLFunctionTypeIntersection));
            if (functionType == MTLFunctionTypeVisible)
            {
                assert(intersectionCompilationMode == IRIntersectionFunctionCompilationVisibleFunction);
                mapping = [[DXR2MTL alloc] initWithName:function.name type:functionType index:visibleFunctions.count];
                [visibleFunctions addObject:function];
            }
            else
            {
                assert(intersectionCompilationMode == IRIntersectionFunctionCompilationIntersectionFunctionBufferFunction);
                mapping = [[DXR2MTL alloc] initWithName:function.name type:functionType index:intersectionFunctions.count];
                [intersectionFunctions addObject:function];
            }
            
            NSString *dxrShaderName0 = anyHitShader.uniqueName;
            convertedMapping[dxrShaderName0] = mapping;
            
            NSString *dxrShaderName1 = intersectionShader.uniqueName;
            convertedMapping[dxrShaderName1] = mapping;
        }
        
        // Closest-hit shader
        if (closestHitShader)
        {
            id<MTLFunction> function = [self _functionWithDXRShader:closestHitShader stage:IRShaderStageClosestHit compiler:compiler error:error];
            if (function == nil)
            {
                return nil;
            }
            
            MTLFunctionType functionType = function.functionType;
            assert(functionType == MTLFunctionTypeVisible);
            
            DXR2MTL *mapping = [[DXR2MTL alloc] initWithName:function.name type:functionType index:visibleFunctions.count];
            [visibleFunctions addObject:function];
            
            NSString *dxrShaderName = closestHitShader.uniqueName;
            convertedMapping[dxrShaderName] = mapping;
        }
    }
    
    // Miss shaders
    for (DXRShader *missShader in descriptor.missShaders)
    {
        id<MTLFunction> function = [self _functionWithDXRShader:missShader stage:IRShaderStageMiss compiler:compiler error:error];
        if (function == nil)
        {
            return nil;
        }
        
        MTLFunctionType functionType = function.functionType;
        assert(functionType == MTLFunctionTypeVisible);
        
        DXR2MTL *mapping = [[DXR2MTL alloc] initWithName:function.name type:functionType index:visibleFunctions.count];
        [visibleFunctions addObject:function];
        
        NSString *dxrShaderName = missShader.uniqueName;
        convertedMapping[dxrShaderName] = mapping;
    }
    
    // Callable shaders
    for (DXRShader *callableShader in descriptor.callableShaders)
    {
        id<MTLFunction> function = [self _functionWithDXRShader:callableShader stage:IRShaderStageMiss compiler:compiler error:error];
        if (function == nil)
        {
            return nil;
        }
        
        MTLFunctionType functionType = function.functionType;
        assert(functionType == MTLFunctionTypeVisible);
        
        DXR2MTL *mapping = [[DXR2MTL alloc] initWithName:function.name type:functionType index:visibleFunctions.count];
        [visibleFunctions addObject:function];
        
        NSString *dxrShaderName = callableShader.uniqueName;
        convertedMapping[dxrShaderName] = mapping;
    }
    
    // Synthesize raygen shader if needed
    if (rayGenCompilationMode == IRRayGenerationCompilationVisibleFunction)
    {
        assert(kernelFunction == nil);
        
        id<MTLFunction> function = [self _synthesizeIndirectRayDispatchKernelWithError:error];
        if (function == nil)
        {
            return nil;
        }
        
        MTLFunctionType functionType = function.functionType;
        assert(functionType == MTLFunctionTypeKernel);
        
        DXR2MTL *mapping = [[DXR2MTL alloc] initWithName:function.name type:functionType];
        [synthesizedMapping addObject:mapping];
        
        kernelFunction = function;
    }
    
    // Synthesize intersection functions if needed
    if (intersectionCompilationMode == IRIntersectionFunctionCompilationVisibleFunction)
    {
        assert(intersectionFunctions.count == 0);
        
        if ((pipelineFlags & IRRaytracingPipelineFlagSkipTriangles) == 0)
        {
            id<MTLFunction> function = [self _synthesizeIntersectionFunctionForHitGroupType:IRHitGroupTypeTriangles error:error];
            if (function == nil)
            {
                return nil;
            }
            
            MTLFunctionType functionType = function.functionType;
            assert(functionType == MTLFunctionTypeIntersection);
            DXR2MTL *mapping = [[DXR2MTL alloc] initWithName:function.name type:functionType index:intersectionFunctions.count];
            [synthesizedMapping addObject:mapping];
            [intersectionFunctions addObject:function];
        }
        
        if ((pipelineFlags & IRRaytracingPipelineFlagSkipProceduralPrimitives) == 0)
        {
            id<MTLFunction> function = [self _synthesizeIntersectionFunctionForHitGroupType:IRHitGroupTypeProceduralPrimitive error:error];
            if (function == nil)
            {
                return nil;
            }
            
            MTLFunctionType functionType = function.functionType;
            assert(functionType == MTLFunctionTypeIntersection);
            DXR2MTL *mapping = [[DXR2MTL alloc] initWithName:function.name type:functionType index:intersectionFunctions.count];
            [synthesizedMapping addObject:mapping];
            [intersectionFunctions addObject:function];
        }
    }
    
    // Create intermediates with mapping info
    DXRIntermediates *intermediates = [[DXRIntermediates alloc] init];
    {
        intermediates.kernelFunction = kernelFunction;
        intermediates.visibleFunctions = [visibleFunctions copy];
        intermediates.intersectionFunctions = [intersectionFunctions copy];
        intermediates.convertedMapping = [convertedMapping copy];
        intermediates.synthesizedMapping = synthesizedMapping.count ? [synthesizedMapping copy] : nil;
    }
    
    return intermediates;
}

- (id<MTLComputePipelineState>)_createComputePipelineStateWithIntermediates:(DXRIntermediates *)intermediates
                                                                 reflection:(MTLComputePipelineReflection **)reflection
                                                                      error:(NSError **)error
{
    id<MTLFunction> dispatchFunction = intermediates.kernelFunction;
    
    NSMutableArray<id<MTLFunction>> *pipelineFunctions = [[NSMutableArray alloc] init];
    [pipelineFunctions addObjectsFromArray:intermediates.visibleFunctions];
    [pipelineFunctions addObjectsFromArray:intermediates.intersectionFunctions];
    
    MTLLinkedFunctions *linkedFunctions = [[MTLLinkedFunctions alloc] init];
    linkedFunctions.functions = [pipelineFunctions copy];
    
    MTLComputePipelineDescriptor *pipelineDescriptor = [[MTLComputePipelineDescriptor alloc] init];
    pipelineDescriptor.computeFunction = dispatchFunction;
    pipelineDescriptor.linkedFunctions = linkedFunctions;
    
    id<MTLComputePipelineState> pipelineState = [_device newComputePipelineStateWithDescriptor:pipelineDescriptor
                                                                                       options:MTLPipelineOptionBindingInfo
                                                                                    reflection:reflection
                                                                                         error:error];
    return pipelineState;
}

- (DXRPipelineState *)_createDXRPipelineStateWithComputePipelineState:(id<MTLComputePipelineState>)computePipelineState
                                                        intermediates:(DXRIntermediates *)intermediates
                                                      compilationMode:(DXRCompilationMode)compilationMode
                                                                error:(NSError **)error
{
    //
    // Create the metal function tables required for the emulating DXR shader tables (depending on the compilation modes).
    // We also need to gather the "shader identifiers" for each metal function that will map to their source DXR shader
    //
    
    id<MTLVisibleFunctionTable> VFTable = nil;
    id<MTLIntersectionFunctionTable> IFTable = nil;
    NSArray<id<MTLFunctionHandle>> *IFBFList = nil;
    
    
    // Use the (function name) + (function type) as a unique key for each identifer
#define _shader_identifier_Key(object) [object.name stringByAppendingString:@(object.functionType).stringValue]
    NSMutableDictionary<NSString *, NSNumber *> *shaderIdentifiers_MTL = [[NSMutableDictionary alloc] init];
    {
        if (intermediates.visibleFunctions)
        {
            // Create VFT (1-based indexing)
            // VFT index 0 is reserved to denote an invalidate shader handle to the metal shader converter runtime.
            NSUInteger startIndex = 1;
            NSUInteger functionCount = intermediates.visibleFunctions.count;
            MTLVisibleFunctionTableDescriptor *vftDescriptor = [[MTLVisibleFunctionTableDescriptor alloc] init];
            vftDescriptor.functionCount = functionCount + startIndex;
            
            id<MTLVisibleFunctionTable> vft = [computePipelineState newVisibleFunctionTableWithDescriptor:vftDescriptor];
            for (NSUInteger i = 0; i < functionCount; ++i)
            {
                uint64_t identifier = i + startIndex;
                id<MTLFunction> function = intermediates.visibleFunctions[i];
                id<MTLFunctionHandle> functionHandle = [computePipelineState functionHandleWithFunction:function];
                [vft setFunction:functionHandle atIndex:identifier];
                
                assert(identifier != DXRShaderIdentifierNull);
                shaderIdentifiers_MTL[_shader_identifier_Key(function)] = @(identifier);
            }
            
            VFTable = vft;
        }
        
        if (intermediates.intersectionFunctions && (compilationMode.intersectionFunction == IRIntersectionFunctionCompilationVisibleFunction))
        {
            // Create IFT (1-based indexing)
            // IFT index 0 is reserved to denote an invalidate shader handle to the metal shader converter runtime.
            NSUInteger startIndex = 1;
            NSUInteger functionCount = intermediates.intersectionFunctions.count;
            MTLIntersectionFunctionTableDescriptor *iftDescriptor = [[MTLIntersectionFunctionTableDescriptor alloc] init];
            iftDescriptor.functionCount = functionCount + startIndex;
            
            id<MTLIntersectionFunctionTable> ift = [computePipelineState newIntersectionFunctionTableWithDescriptor:iftDescriptor];
            for (NSUInteger i = 0; i < functionCount; ++i)
            {
                uint64_t identifier = i + startIndex;
                id<MTLFunction> function = intermediates.intersectionFunctions[i];
                id<MTLFunctionHandle> functionHandle = [computePipelineState functionHandleWithFunction:function];
                [ift setFunction:functionHandle atIndex:identifier];
                
                assert(identifier != DXRShaderIdentifierNull);
                shaderIdentifiers_MTL[_shader_identifier_Key(function)] = @(identifier);
            }
            
            IFTable = ift;
        }
        
        if (intermediates.intersectionFunctions && (compilationMode.intersectionFunction == IRIntersectionFunctionCompilationIntersectionFunctionBufferFunction))
        {
            // Create interesction function handles
            // For IRIntersectionFunctionCompilationIntersectionFunctionBufferFunction, the shader identifier must be the resource id of the function handle
            NSMutableArray<id<MTLFunctionHandle>> *intersectionFunctionHandles = [[NSMutableArray alloc] init];
            
            for (id<MTLFunction> function in intermediates.intersectionFunctions)
            {
                id<MTLFunctionHandle> functionHandle = [computePipelineState functionHandleWithFunction:function];
                [intersectionFunctionHandles addObject:functionHandle];
                
                uint64_t identifier = functionHandle.gpuResourceID._impl;
                assert(identifier != DXRShaderIdentifierNull);
                shaderIdentifiers_MTL[_shader_identifier_Key(function)] = @(identifier);
            }
            
            IFBFList = [intersectionFunctionHandles copy];
        }
        
        if (intermediates.kernelFunction)
        {
            // This is not required but since `0` is being reserved to denote "invalid shader handle", assign a non-zero identifier.
            id<MTLFunction> function = intermediates.kernelFunction;
            shaderIdentifiers_MTL[_shader_identifier_Key(function)] = @(UINT16_MAX);
        }
    }
    
    // Now use the mapping info from 'DXRIntermediates' to map identifiers to the source DXR shaders
    NSMutableDictionary<NSString *, NSNumber *> *shaderIdentifiers_DXR = [[NSMutableDictionary alloc] init];
    {
        for (NSString *shaderUniqueName in intermediates.convertedMapping)
        {
            id<MTLFunction> function = nil;
            DXR2MTL *mapping = intermediates.convertedMapping[shaderUniqueName];
            
            if (mapping.functionType == MTLFunctionTypeKernel)
            {
                function = intermediates.kernelFunction;
            }
            else if (mapping.functionType == MTLFunctionTypeIntersection)
            {
                NSUInteger locationIndex = mapping.locationIndex;
                function = intermediates.intersectionFunctions[locationIndex];
            }
            else if (mapping.functionType == MTLFunctionTypeVisible)
            {
                NSUInteger locationIndex = mapping.locationIndex;
                function = intermediates.visibleFunctions[locationIndex];
            }
            
            if (function)
            {
                shaderIdentifiers_DXR[shaderUniqueName] = shaderIdentifiers_MTL[_shader_identifier_Key(function)];
            }
        }
        
        for (DXR2MTL *mapping in intermediates.synthesizedMapping)
        {
            id<MTLFunction> function = nil;
            
            if (mapping.functionType == MTLFunctionTypeKernel)
            {
                function = intermediates.kernelFunction;
            }
            else if (mapping.functionType == MTLFunctionTypeIntersection)
            {
                NSUInteger locationIndex = mapping.locationIndex;
                function = intermediates.intersectionFunctions[locationIndex];
            }
            else if (mapping.functionType == MTLFunctionTypeVisible)
            {
                NSUInteger locationIndex = mapping.locationIndex;
                function = intermediates.visibleFunctions[locationIndex];
            }
            
            if (function)
            {
                shaderIdentifiers_DXR[function.name] = shaderIdentifiers_MTL[_shader_identifier_Key(function)];
            }
        }
    }
    
    //
    // Create the DXR pipeline state object
    //
    
    DXRPipelineState *pipelineState = [[DXRPipelineState alloc] init];
    
    // the core ray tracing metal compute pipeline
    pipelineState.computePipelineState = computePipelineState;
    
    // the resident ray tracing resources
    pipelineState.visibleFunctionTable = VFTable;
    pipelineState.intersectionFunctionTable = IFTable;
    pipelineState.intersectionFunctionBufferHandles = IFBFList;
    
    // objects to help determine valid shader identifiers
    pipelineState.shaderIdentifiers = [shaderIdentifiers_DXR copy];
    
    return pipelineState;
}

- (DXRPipelineReflection *)_createReflectionWithComputePipelineReflection:(MTLComputePipelineReflection *)computePipelineReflection
                                                            intermediates:(DXRIntermediates *)intermediates
{
    NSMutableArray<DXRShaderConversion *> *conversions = [[NSMutableArray alloc] init];
    for (NSString *shaderUniqueName in intermediates.convertedMapping)
    {
        id<MTLFunction> function = nil;
        DXR2MTL *mapping = intermediates.convertedMapping[shaderUniqueName];
        
        if (mapping.functionType == MTLFunctionTypeKernel)
        {
            function = intermediates.kernelFunction;
        }
        else if (mapping.functionType == MTLFunctionTypeIntersection)
        {
            NSUInteger locationIndex = mapping.locationIndex;
            function = intermediates.intersectionFunctions[locationIndex];
        }
        else if (mapping.functionType == MTLFunctionTypeVisible)
        {
            NSUInteger locationIndex = mapping.locationIndex;
            function = intermediates.visibleFunctions[locationIndex];
        }
        
        if (function)
        {
            DXRShaderConversion *conversion = [[DXRShaderConversion alloc] initWithName:shaderUniqueName function:function isSynthesized:NO];
            [conversions addObject:conversion];
        }
    }
    
    for (DXR2MTL *mapping in intermediates.synthesizedMapping)
    {
        id<MTLFunction> function = nil;
        
        if (mapping.functionType == MTLFunctionTypeKernel)
        {
            function = intermediates.kernelFunction;
        }
        else if (mapping.functionType == MTLFunctionTypeIntersection)
        {
            NSUInteger locationIndex = mapping.locationIndex;
            function = intermediates.intersectionFunctions[locationIndex];
        }
        else if (mapping.functionType == MTLFunctionTypeVisible)
        {
            NSUInteger locationIndex = mapping.locationIndex;
            function = intermediates.visibleFunctions[locationIndex];
        }
        
        if (function)
        {
            DXRShaderConversion *conversion = [[DXRShaderConversion alloc] initWithName:@"" function:function isSynthesized:YES];
            [conversions addObject:conversion];
        }
    }
    
    DXRPipelineReflection *pipelineReflection = [[DXRPipelineReflection alloc] init];
    pipelineReflection.computePipelineReflection = computePipelineReflection;
    pipelineReflection.conversions = [conversions copy];
    
    return pipelineReflection;
}

- (id<MTLFunction>)_synthesizeIndirectRayDispatchKernelWithError:(NSError **)error
{
    __attribute__((__cleanup__(destroy_ir_compiler)))
    IRCompiler *compiler = DefaultIRCompilerCreate(self.gpuFamily);
    
    __attribute__((__cleanup__(destroy_ir_metallib)))
    IRMetalLibBinary *metalLib  = IRMetalLibBinaryCreate();
    
    if (!IRMetalLibSynthesizeIndirectRayDispatchFunction(compiler, metalLib))
    {
        out_assign(error, default_error(@"Error synthesizing indirect ray dispatch function"));
        return nil;
    }
    
    id<MTLLibrary> library = [_device newLibraryWithData:IRMetalLibGetBytecodeData(metalLib) error:error];
    if (library == nil)
    {
        return nil;
    }
    [library setLabel:@"SynthesizedRayDispatchKernel"];
    
    id<MTLFunction> function = [library newFunctionWithName:@(kIRRayDispatchIndirectionKernelName)];
    if (function == nil)
    {
        out_assign(error, default_error(@"Error creating synthesized dispatch function"));
    }
    
    return function;
}

- (id<MTLFunction>)_synthesizeIntersectionFunctionForHitGroupType:(IRHitGroupType)hitGroupType error:(NSError **)error
{
    __attribute__((__cleanup__(destroy_ir_compiler)))
    IRCompiler *compiler = DefaultIRCompilerCreate(self.gpuFamily);
    IRCompilerSetHitgroupType(compiler, hitGroupType);
    
    __attribute__((__cleanup__(destroy_ir_metallib)))
    IRMetalLibBinary *metalLib = IRMetalLibBinaryCreate();
    if (!IRMetalLibSynthesizeIndirectIntersectionFunction(compiler, metalLib))
    {
        out_assign(error, default_error(@"Error synthesizing intersection function"));
        return nil;
    }
    
    id<MTLLibrary> library = [_device newLibraryWithData:IRMetalLibGetBytecodeData(metalLib) error:error];
    if (library == nil)
    {
        return nil;
    }
    
    BOOL isHgTriangles = (hitGroupType == IRHitGroupTypeTriangles);
    [library setLabel:isHgTriangles ? @"SynthesizedTrianglesIntersectionFunction" : @"SynthesizedProceduralIntersectionFunction"];
    
    NSString *functionName = isHgTriangles ? @(kIRIndirectTriangleIntersectionFunctionName) : @(kIRIndirectProceduralIntersectionFunctionName);
    id<MTLFunction> function = [library newFunctionWithName:functionName];
    if (function == nil)
    {
        out_assign(error, default_error(@"Error creating synthesized dispatch function"));
    }
    
    return function;
}

- (id<MTLFunction>)_functionWithDXRShader:(DXRShader *)shader stage:(IRShaderStage)stage compiler:(IRCompiler *)compiler error:(NSError **)error
{
    return [self _functionWithDXRShader:shader stage:stage fuseWithShader:nil compiler:compiler error:error];
}

- (id<MTLFunction>)_functionWithDXRAnyHitShader:(DXRShader *)anyHitShader fuseedWithIntersectionShader:(DXRShader *)intersectionShader compiler:(IRCompiler *)compiler error:(NSError **)error
{
    return [self _functionWithDXRShader:anyHitShader stage:IRShaderStageAnyHit fuseWithShader:intersectionShader compiler:compiler error:error];
}

- (id<MTLFunction>)_functionWithDXRShader:(DXRShader *)shader stage:(IRShaderStage)stage fuseWithShader:(nullable DXRShader *)fuseShader compiler:(IRCompiler *)compiler error:(NSError **)error
{
    __attribute__((__cleanup__(destroy_ir_object)))
    IRObject *AIR = NULL;
    const char *name = shader.name.UTF8String;
    const char *uniqueName = shader.uniqueName.UTF8String;
    
    // Create DXIL bytecode object (but keep memory ownership with shader.DXIL)
    IRObject *DXIR = IRObjectCreateFromDXIL(shader.DXIL.bytes, shader.DXIL.length, IRBytecodeOwnershipNone);
    if (!DXIR)
    {
        NSString *description = [NSString stringWithFormat:@"Could not load DXIL data for shader \"%s\"", uniqueName];
        out_assign(error, default_error(description));
        return nil;
    }
    
    IRError *compileError = NULL;
    IRCompilerSetEntryPointName(compiler, uniqueName);
    
    if (fuseShader)
    {
        // Fusing is only allowed between intersection and anyHit shaders
        // metal shader converter requires the DXIL data for both shaders to be the same
        const char *intersectionFunctionName = fuseShader.name.UTF8String;
        const char *anyHitFunctionName = name;
        AIR = IRCompilerAllocCombineCompileAndLink(compiler, intersectionFunctionName, DXIR, anyHitFunctionName, DXIR, &compileError);
    }
    else
    {
        AIR = IRCompilerAllocCompileAndLink(compiler, name, DXIR, &compileError);
    }
    IRObjectDestroy(DXIR);
    
    if (!AIR)
    {
        NSString *description = [NSString stringWithFormat:@"Failed to convert shader \"%s\": %s", uniqueName, (const char*)IRErrorGetPayload(compileError)];
        out_assign(error, default_error(description));
        IRErrorDestroy(compileError);
        return nil;
    }
    
    __attribute__((__cleanup__(destroy_ir_metallib)))
    IRMetalLibBinary *metalLib = IRMetalLibBinaryCreate();
    if (!IRObjectGetMetalLibBinary(AIR, stage, metalLib))
    {
        NSString *description = [NSString stringWithFormat:@"Failed to create MetalLib for shader \"%s\"", uniqueName];
        out_assign(error, default_error(description));
        return nil;
    }
    
    NSError *libraryError;
    id<MTLLibrary> library = [_device newLibraryWithData:IRMetalLibGetBytecodeData(metalLib) error:&libraryError];
    if (library == nil)
    {
        NSString *description = [NSString stringWithFormat:@"Failed to create MTLLibrary for shader \"%s\": %@", uniqueName, libraryError.localizedDescription];
        out_assign(error, default_error(description));
        return nil;
    }
    [library setLabel:@(uniqueName)];
    
    // Determine the newly created function name
    // Since these are single shader compilations, pulling the name from library is safest way to get the converted metal function name
    // For any-hit and intersection shaders, the function name may change during linking - likely as a result of fusing with an intersection function
    // either explicity with fuseShader or implicitly with metal shader converter's synthesized functions
    NSString *functionName = library.functionNames.firstObject;

    // If library has multiple functions (possible with offline compilation), querying the IR reflection should do the trick
    // For any-hit and intersection shaders, a selection heuristic will be needed instead.
    if ((library.functionNames.count > 1) && ![functionName containsString:@(uniqueName)])
    {
        if ((stage == IRShaderStageAnyHit) || (stage == IRShaderStageIntersection))
        {
            for (NSString *n in library.functionNames) {
                if ([n containsString:@(uniqueName)]) {
                    functionName = n;
                    break;
                }
            }
        }
        else
        {
            IRShaderReflection *reflection = IRShaderReflectionCreate();
            if (IRObjectGetReflection(AIR, stage, reflection))
            {
                const char *newName = IRShaderReflectionGetEntryPointFunctionName(reflection);
                functionName = newName ? @(newName) : nil;
                IRShaderReflectionDestroy(reflection);
            }
        }
    }
    
    assert(functionName.length != 0);
    id<MTLFunction> function = functionName.length ? [library newFunctionWithName:functionName] : nil;
    if (function == nil)
    {
        NSString *description = [NSString stringWithFormat:@"Failed to create MTLFunction named \"%@\" for shader \"%s\"", functionName, uniqueName];
        out_assign(error, default_error(description));
        return nil;
    }
    
    return function;
}

- (IRGPUFamily)gpuFamily
{
    if ([_device supportsFamily:MTLGPUFamilyApple9]) {
        return IRGPUFamilyApple9;
    }
    
    if ([_device supportsFamily:MTLGPUFamilyApple8]) {
        return IRGPUFamilyApple8;
    }
    
    if ([_device supportsFamily:MTLGPUFamilyApple7]) {
        return IRGPUFamilyApple7;
    }
    
    return IRGPUFamilyApple6;
}
@end

@implementation DXRPipelineState
- (uint64_t)shaderIdentifierForName:(nonnull NSString *)name
{
    NSNumber *number = self.shaderIdentifiers[name];
    if (number == nil)
    {
        return DXRShaderIdentifierNull;
    }
    
    uint64_t shaderIdentifier = number.unsignedLongLongValue;
    return shaderIdentifier;
}

- (uint64_t)shaderIdentifierForSynthesizedTriangleIntersection
{
    // If compiled with IRIntersectionFunctionCompilationModeVisibleFunction, and pipeline has a hitgroup of type IRHitGroupTypeTriangles
    // it is expected to contain the synthesized indirect triangle intersection function
    return [self shaderIdentifierForName:@(kIRIndirectTriangleIntersectionFunctionName)];
}

- (uint64_t)shaderIdentifierForSynthesizedProceduralIntersection
{
    // If compiled with IRIntersectionFunctionCompilationModeVisibleFunction, and pipeline has a hitgroup of type IRHitGroupTypeProcedural
    // it is expected to contain the synthesized indirect procedural intersection function
    return [self shaderIdentifierForName:@(kIRIndirectProceduralIntersectionFunctionName)];
}

@end

@implementation DXRShader
- (instancetype)initWithName:(NSString *)name DXIL:(NSData *)DXIL
{
    self = [super init];
    if (self)
    {
        self->_name = name;
        self->_DXIL = DXIL;
    }
    return self;
}

- (NSString *)uniqueName
{
    if (self->_specializedName != nil)
    {
        return _specializedName;
    }
    return _name;
}
@end

@implementation DXRHitgroup
- (instancetype)initWithName:(NSString *)name
          intersectionShader:(nullable DXRShader *)intersectionShader
                anyHitShader:(nullable DXRShader *)anyHitShader
            closestHitShader:(nullable DXRShader *)closestHitShader
{
    self = [super init];
    if (self)
    {
        self->_name = name;
        self->_intersectionShader = intersectionShader;
        self->_anyHitShader = anyHitShader;
        self->_closestHitShader = closestHitShader;
    }
    return self;
}

@end

@implementation DXRShaderConversion
- (instancetype)initWithName:(NSString *)name function:(id<MTLFunction>)function isSynthesized:(BOOL)isSynthesized
{
    self = [super init];
    if (self)
    {
        self->_name = name;
        self->_function = function;
        self->_isSynthesized = isSynthesized;
    }
    return self;
}
@end

@implementation DXR2MTL
- (instancetype)initWithName:(NSString *)name type:(MTLFunctionType)functionType index:(NSUInteger)index
{
    self = [super init];
    if (self)
    {
        self->_name = name;
        self->_functionType = functionType;
        self->_locationIndex = index;
    }
    return self;
}
- (instancetype)initWithName:(NSString *)name type:(MTLFunctionType)functionType
{
    return [self initWithName:name type:functionType index:0];
}
@end

@implementation DXRPipelineOptions
- (instancetype)init
{
    self = [super init];
    if (self)
    {
        // Metal shader converter defaults
        self->_maxRecursiveDepth = IRRayTracingUnlimitedRecursionDepth;
        self->_maxAttributeSizeInBytes = 16;
    }
    return self;
}
@end

@implementation DXRIntermediates
@end

@implementation DXRPipelineDescriptor
@end

@implementation DXRPipelineReflection
@end


